# Copyright 2011 iXsystems, Inc.
# All rights reserved
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted providing that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#
#####################################################################
import logging
import os
import re

from django.core.urlresolvers import reverse
from django.forms.widgets import Widget, TextInput
from django.forms.utils import flatatt
from django.template.loader import render_to_string
from django.utils.safestring import mark_safe
from django.utils.encoding import force_str
from django.utils.translation import ugettext_lazy as _

from dojango import forms
from dojango.forms import widgets
from dojango.forms.widgets import DojoWidgetMixin
from freenasUI.account.models import bsdGroups, bsdUsers
from freenasUI.storage.models import Volume
from freenasUI.middleware.client import client

import ipaddr

MAC_RE = re.compile(r'^[0-9A-F]{12}$')

log = logging.getLogger('freeadmin.forms')


class CronMultiple(DojoWidgetMixin, Widget):
    dojo_type = 'freeadmin.form.Cron'

    def render(self, name, value, attrs=None):
        if value is None:
            value = ''
        final_attrs = self.build_attrs(
            attrs, name=name, **{'data-dojo-name': name}
        )
        final_attrs['value'] = force_str(value)
        if value.startswith('*/'):
            final_attrs['typeChoice'] = "every"
        elif re.search(r'^[0-9].*', value):
            final_attrs['typeChoice'] = "selected"
        return mark_safe('<div%s></div>' % (flatatt(final_attrs),))


class DirectoryBrowser(TextInput):
    def __init__(self, *args, **kwargs):
        dirsonly = kwargs.pop('dirsonly', True)
        kwargs.pop('filesonly', False)
        super(DirectoryBrowser, self).__init__(*args, **kwargs)
        self.attrs.update({
            'dojoType': 'freeadmin.form.PathSelector',
            'dirsonly': str(dirsonly),
        })


class UserField(forms.ChoiceField):
    widget = widgets.Select()

    def __init__(self, *args, **kwargs):
        kwargs.pop('max_length', None)
        self._exclude = kwargs.pop('exclude', [])
        super(UserField, self).__init__(*args, **kwargs)

    def prepare_value(self, value):
        rv = super(UserField, self).prepare_value(value)
        if rv:
            try:
                with client as c:
                    c.call('dscache.get_uncached_user', rv)
            except:
                log.warn('Failed to get user', exc_info=True)
                rv = 'nobody'
        return rv

    def _reroll(self):
        from freenasUI.account.forms import FilteredSelectJSON
        try:
            with client as c:
                users = c.call('users.query', {'extra': {'search_dscache': True}})
        except:
            users = []
        kwargs = {
            'api_name': 'v1.0',
            'resource_name': 'account/all_users'
        }
        if len(users) > 500:
            if self.initial:
                self.choices = ((self.initial, self.initial),)
            if len(self._exclude) > 0:
                kwargs['exclude'] = ','.join(self._exclude)
            self.widget = FilteredSelectJSON(
                attrs=self.widget.attrs,
                url=reverse('api_dispatch_list', kwargs=kwargs)
            )
        else:
            ulist = []
            if not self.required:
                ulist.append(('-----', 'N/A'))
            notbuiltin = [
                o[0]
                for o in bsdUsers.objects.filter(
                    bsdusr_builtin=False
                ).values_list('bsdusr_uid')
            ]
            ulist.extend(
                [(x['username'], x['username'] ) for x in sorted([y for y in users if (
                            y is not None and y['username'] not in self._exclude
                        )], key=lambda y: (y['uid'] not in notbuiltin, y['username']))]
            )

            self.widget = FilteredSelectJSON(
                attrs=self.widget.attrs,
                url=reverse('api_dispatch_list', kwargs=kwargs),
                choices=ulist
            )
            #self.choices = ulist

    def clean(self, user):
        if not self.required and user in ('-----', '', None):
            return None
        try:
            with client as c:
                u = c.call('dscache.get_uncached_user', user)
        except:
            log.warn('Failed to get user', exc_info=True)
            u = None

        if u is None:
            raise forms.ValidationError(_("The user %s is not valid.") % user)
        return user


class GroupField(forms.ChoiceField):
    widget = widgets.Select()

    def __init__(self, *args, **kwargs):
        kwargs.pop('max_length', None)
        super(GroupField, self).__init__(*args, **kwargs)

    def prepare_value(self, value):
        rv = super(GroupField, self).prepare_value(value)
        if rv:
            try:
                with client as c:
                    g = c.call('dscache.get_uncached_group', rv)
            except:
                rv = 'nobody'
        return rv

    def _reroll(self):
        from freenasUI.account.forms import FilteredSelectJSON
        try:
            with client as c:
                groups = c.call('groups.query', {'extra': {'search_dscache': True}})
        except:
            groups = []
        kwargs = {
            'api_name': 'v1.0',
            'resource_name': 'account/all_groups'
        }
        if len(groups) > 500:
            if self.initial:
                self.choices = ((self.initial, self.initial),)
            self.widget = FilteredSelectJSON(
                attrs=self.widget.attrs,
                url=reverse('api_dispatch_list', kwargs=kwargs)
            )
        else:
            glist = []
            if not self.required:
                glist.append(('-----', 'N/A'))
            notbuiltin = [
                o[0]
                for o in bsdGroups.objects.filter(
                    bsdgrp_builtin=False
                ).values_list('bsdgrp_gid')
            ]
            glist.extend(
                [(x['group'], x['group']) for x in sorted(
                    groups,
                    key=lambda y: (y['gid'] not in notbuiltin, y['group'])
                )]
            )
            #self.choices = glist
            self.widget = FilteredSelectJSON(
                attrs=self.widget.attrs,
                url=reverse('api_dispatch_list', kwargs=kwargs),
                choices=glist
            )

    def clean(self, group):
        if not self.required and group in ('-----', '', None):
            return None
        try:
            with client as c:
                g = c.call('dscache.get_uncached_group', group)
        except:
            g = None

        if g is None:
            raise forms.ValidationError(_("The group %s is not valid.") % group)
        return group


class PathField(forms.CharField):

    def __init__(self, *args, **kwargs):
        self.dirsonly = kwargs.pop('dirsonly', True)
        self.filesonly = kwargs.pop('filesonly', False)
        if self.dirsonly and self.filesonly:
            raise ValueError("You cannot have dirsonly _and_ filesonly")
        self.abspath = kwargs.pop('abspath', True)
        self.includes = kwargs.pop('includes', [])
        self.widget = DirectoryBrowser(
            dirsonly=str(self.dirsonly),
            filesonly=str(self.filesonly)
        )
        super(PathField, self).__init__(*args, **kwargs)

    def clean(self, value):
        if value not in ('', None):
            value = value.strip()
            absv = os.path.abspath(value)
            valid = False
            for v in Volume.objects.all().values_list('vol_name',):
                path = '/mnt/%s' % v[0]
                if absv.startswith(path + '/') or absv == path:
                    valid = True
                    break
            if not valid and absv in self.includes:
                valid = True
            if not valid:
                raise forms.ValidationError(
                    _("The path must reside within a volume mount point")
                )
            if self.filesonly:
                if not(
                    os.path.exists(absv) and
                    (os.path.isfile(value) or os.path.islink(value))
                ):
                    raise forms.ValidationError(
                        _("A file is required")
                    )
            return value if not self.abspath else absv
        return value


class MACField(forms.CharField):

    def __init__(self, *args, **kwargs):
        kwargs['max_length'] = 17
        super(MACField, self).__init__(*args, **kwargs)

    def clean(self, value):
        value = value.upper().replace(':', '')
        if value and not MAC_RE.search(value):
            raise forms.ValidationError("%s is not a valid MAC Address" % (
                value,
            ))
        value = super(MACField, self).clean(value)
        return value


class SizeWidget(widgets.TextInput):
    def format_value(self, value):
        if isinstance(value, str):
            return value

        for suffix, threshold in sorted(SizeField.SUFFIXES.items(), key=lambda item: -item[1]):
            if value >= threshold:
                return ("%.3f" % (value / threshold)).rstrip("0").rstrip(".") + " " + suffix + ("iB" if suffix != "B"
                                                                                                else "")

        return "%d B" % value


class SizeField(forms.CharField):

    SUFFIXES = {
        'P': 1125899906842624,
        'T': 1099511627776,
        'G': 1073741824,
        'M': 1048576,
        'K': 1024,
        'B': 1,
    }

    widget = SizeWidget

    def to_python(self, value):
        value = str(value)
        if not self.required and value.strip() in ['', '0']:
            return 0

        value = value.replace(' ', '')
        reg = re.search(r'^(\d+(?:\.\d+)?)([BKMGTP](?:iB)?)$', value, re.I)
        if not reg:
            raise forms.ValidationError(
                _('Specify the value with IEC suffixes, e.g. 10 GiB')
            )

        number, suffix = reg.groups()

        number = int(float(number) * self.SUFFIXES[suffix[0].upper()])

        return number


class SelectMultipleWidget(forms.widgets.SelectMultiple):

    def __init__(self, attrs=None, choices=(), sorter=False):
        self._sorter = sorter
        super(SelectMultipleWidget, self).__init__(attrs, choices)

    def render(self, name, value, attrs=None):

        if value is None:
            value = []
        selected = []
        unselected = []
        for choice in list(self.choices):
            if choice[0] not in value:
                unselected.append(choice)

        for v in value:
            for choice in list(self.choices):
                if v == choice[0]:
                    selected.append(choice)
                    break

        select_available = forms.widgets.SelectMultiple(
            attrs={'size': 6}, choices=unselected
        ).render(
            'selecAt_from', value, {'id': 'select_from'},
        )
        select_available = ''.join(select_available.split('</select>')[:-1])
        select_selected = forms.widgets.SelectMultiple(
            attrs={'size': 6}, choices=selected
        ).render(
            name, value, attrs,
        )
        select_selected = ''.join(select_selected.split('</select>')[:-1])
        output = render_to_string('freeadmin/selectmultiple.html', {
            'attrs': attrs,
            'select_available': select_available,
            'select_selected': select_selected,
            'fromid': 'select_from',
            'sorter': self._sorter,
        })
        return output


class SelectMultipleField(forms.fields.MultipleChoiceField):
    widget = SelectMultipleWidget

    def __init__(self, *args, **kwargs):
        super(SelectMultipleField, self).__init__(*args, **kwargs)


class Network4Field(forms.CharField):

    def __init__(self, *args, **kwargs):
        kwargs['max_length'] = 18  # 255.255.255.255/32
        super(Network4Field, self).__init__(*args, **kwargs)

    def clean(self, value):
        if not value:
            return value
        try:
            value = str(ipaddr.IPv4Network(value))
        except ipaddr.AddressValueError as e:
            raise forms.ValidationError(
                _("Invalid address: %s") % e
            )
        except ipaddr.NetmaskValueError as e:
            raise forms.ValidationError(
                _("Invalid network: %s") % e
            )
        value = super(Network4Field, self).clean(value)
        return value


class Network6Field(forms.CharField):

    def __init__(self, *args, **kwargs):
        # ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff/128
        kwargs['max_length'] = 43
        super(Network6Field, self).__init__(*args, **kwargs)

    def clean(self, value):
        if not value:
            return value
        try:
            value = str(ipaddr.IPv6Network(value))
        except ipaddr.AddressValueError as e:
            raise forms.ValidationError(
                _("Invalid address: %s") % e
            )
        except ipaddr.NetmaskValueError as e:
            raise forms.ValidationError(
                _("Invalid network: %s") % e
            )
        value = super(Network6Field, self).clean(value)
        return value


class WarningWidgetMixin(object):
    """
    This mixin for widgets adds a warning text above the widget

    The argument text is taken for that purpose
    """
    def __init__(self, *args, **kwargs):
        self.text = kwargs.pop('text')
        super(WarningWidgetMixin, self).__init__(*args, **kwargs)

    def render(self, *args, **kwargs):
        rendered = super(WarningWidgetMixin, self).render(*args, **kwargs)
        return "%s<br />\n%s" % (self.text, rendered)


class WarningSelect(WarningWidgetMixin, forms.widgets.Select):
    pass
